<?php

use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\BoostController;
use App\Http\Controllers\Api\BrandController;
use App\Http\Controllers\Api\CampaignController;
use App\Http\Controllers\Api\ChatController;
use App\Http\Controllers\Api\CreatorController;
use App\Http\Controllers\Api\FeedController;
use App\Http\Controllers\Api\NotificationController;
use App\Http\Controllers\Api\PaymentController;
use App\Http\Controllers\Api\SyncController;
use App\Http\Controllers\Api\UploadController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Admin\AdminController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| All routes are prefixed with /api/v1 (configured in bootstrap/app.php)
|
*/

// ===========================================
// AUTHENTICATION ROUTES (Public)
// ===========================================

Route::prefix('auth')->group(function () {
    Route::post('/request-otp', [AuthController::class, 'requestOtp']);
    Route::post('/verify-otp', [AuthController::class, 'verifyOtp']);
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/refresh', [AuthController::class, 'refreshToken']);
    Route::post('/admin/login', [AuthController::class, 'adminLogin']);
    Route::post('/logout', [AuthController::class, 'logout']);
});

// ===========================================
// USER ROUTES (Protected)
// ===========================================

Route::prefix('users')->middleware('auth:sanctum')->group(function () {
    Route::get('/me', [UserController::class, 'getProfile']);
    Route::put('/me', [UserController::class, 'updateProfile']);
    Route::put('/fcm-token', [UserController::class, 'updateFcmToken']);
    Route::put('/onboarding-complete', [UserController::class, 'completeOnboarding']);
    Route::delete('/me', [UserController::class, 'deactivateAccount']);
});

// ===========================================
// BRAND ROUTES
// ===========================================

Route::prefix('brands')->group(function () {
    // Public routes
    Route::get('/', [BrandController::class, 'index']);
    Route::get('/{id}', [BrandController::class, 'show']);

    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::get('/profile', [BrandController::class, 'getProfile']);
        Route::put('/profile', [BrandController::class, 'updateProfile']);
        Route::get('/wallet', [BrandController::class, 'getWallet']);
        Route::get('/transactions', [BrandController::class, 'getTransactions']);
    });
});

// ===========================================
// CREATOR ROUTES
// ===========================================

Route::prefix('creators')->group(function () {
    // Public routes
    Route::get('/', [CreatorController::class, 'index']);
    Route::get('/featured', [CreatorController::class, 'featured']);
    Route::get('/{id}', [CreatorController::class, 'show']);

    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::get('/profile', [CreatorController::class, 'getProfile']);
        Route::put('/profile', [CreatorController::class, 'updateProfile']);
        Route::put('/portfolio', [CreatorController::class, 'updatePortfolio']);
        Route::put('/rate-card', [CreatorController::class, 'updateRateCard']);
        Route::get('/wallet', [CreatorController::class, 'getWallet']);
        Route::get('/transactions', [CreatorController::class, 'getTransactions']);
    });
});

// ===========================================
// CAMPAIGN ROUTES
// ===========================================

Route::prefix('campaigns')->group(function () {
    // Public routes
    Route::get('/', [CampaignController::class, 'index']);
    Route::get('/featured', [CampaignController::class, 'featured']);
    Route::get('/{id}', [CampaignController::class, 'show']);

    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('/', [CampaignController::class, 'store']);
        Route::put('/{id}', [CampaignController::class, 'update']);
        Route::delete('/{id}', [CampaignController::class, 'destroy']);
        Route::get('/{id}/applications', [CampaignController::class, 'applications']);
        Route::post('/{id}/invite', [CampaignController::class, 'invite']);
        Route::post('/{id}/apply', [CampaignController::class, 'apply']);
    });
});

// ===========================================
// SYNC ROUTES (Protected)
// ===========================================

Route::prefix('syncs')->middleware('auth:sanctum')->group(function () {
    Route::get('/', [SyncController::class, 'index']);
    Route::get('/{id}', [SyncController::class, 'show']);
    Route::put('/{id}', [SyncController::class, 'update']);
    Route::put('/{id}/status', [SyncController::class, 'updateStatus']);
    Route::post('/{id}/submit', [SyncController::class, 'submitContent']);
    Route::post('/{id}/approve', [SyncController::class, 'approveContent']);
    Route::post('/{id}/request-revision', [SyncController::class, 'requestRevision']);
    Route::post('/{id}/complete', [SyncController::class, 'complete']);
});

// ===========================================
// CHAT ROUTES (Protected)
// ===========================================

Route::prefix('chats')->middleware('auth:sanctum')->group(function () {
    Route::get('/', [ChatController::class, 'index']);
    Route::get('/{id}', [ChatController::class, 'show']);
    Route::get('/{id}/messages', [ChatController::class, 'getMessages']);
    Route::post('/{id}/messages', [ChatController::class, 'sendMessage']);
    Route::put('/{id}/read', [ChatController::class, 'markAsRead']);
});

// ===========================================
// NOTIFICATION ROUTES (Protected)
// ===========================================

Route::prefix('notifications')->middleware('auth:sanctum')->group(function () {
    Route::get('/', [NotificationController::class, 'index']);
    Route::get('/unread-count', [NotificationController::class, 'getUnreadCount']);
    Route::put('/{id}/read', [NotificationController::class, 'markAsRead']);
    Route::put('/read-all', [NotificationController::class, 'markAllAsRead']);
});

// ===========================================
// UPLOAD ROUTES (Protected)
// ===========================================

Route::prefix('uploads')->middleware('auth:sanctum')->group(function () {
    Route::post('/presigned-url', [UploadController::class, 'getPresignedUrl']);
    Route::post('/confirm', [UploadController::class, 'confirmUpload']);
    Route::post('/image', [UploadController::class, 'uploadImage']);
    Route::post('/admin/asset', [UploadController::class, 'uploadAdminAsset']);
    Route::delete('/{id}', [UploadController::class, 'destroy']);
    Route::get('/admin/assets', [UploadController::class, 'listAssets']); // Need to implement this too
});

// ===========================================
// BOOST ROUTES
// ===========================================

Route::prefix('boosts')->group(function () {
    // Public routes
    Route::get('/packages', [BoostController::class, 'listPackages']);
    Route::get('/packages/{id}', [BoostController::class, 'showPackage']);

    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('/purchase', [BoostController::class, 'purchase']);
        Route::get('/active', [BoostController::class, 'getActiveBoosts']);
        Route::get('/history', [BoostController::class, 'getHistory']);
    });
});

// ===========================================
// FEED ROUTES
// ===========================================

Route::prefix('feed')->group(function () {
    // Public routes (with optional auth for liked status)
    Route::get('/', [FeedController::class, 'index']);
    Route::get('/{id}', [FeedController::class, 'show']);

    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('/', [FeedController::class, 'store']);
        Route::post('/{id}/like', [FeedController::class, 'like']);
        Route::delete('/{id}/like', [FeedController::class, 'unlike']);
        Route::delete('/{id}', [FeedController::class, 'destroy']);
    });
});

// ===========================================
// PAYMENT ROUTES
// ===========================================

Route::prefix('payments')->group(function () {
    // Webhook (no auth required)
    Route::post('/webhook', [PaymentController::class, 'handleWebhook']);

    // User payment routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('/create-order', [PaymentController::class, 'createOrder']);
        Route::post('/verify', [PaymentController::class, 'verifyPayment']);
    });

    // Admin payment routes
    Route::prefix('admin')->middleware(['auth:sanctum', 'admin'])->group(function () {
        Route::get('/transactions', [PaymentController::class, 'listAllTransactions']);
        Route::get('/escrow', [PaymentController::class, 'getEscrowSummary']);
        Route::post('/brands/{id}/adjust-balance', [PaymentController::class, 'adjustBrandBalance']);
        Route::post('/creators/{id}/adjust-balance', [PaymentController::class, 'adjustCreatorBalance']);
    });
});

// ===========================================
// ADMIN ROUTES (Admin only)
// ===========================================

Route::prefix('admin')->middleware(['auth:sanctum', 'admin'])->group(function () {
    // Dashboard
    Route::get('/dashboard', [AdminController::class, 'getDashboardStats']);
    Route::get('/analytics', [AdminController::class, 'getAnalytics']);

    // Entity Management
    Route::get('/brands', [AdminController::class, 'listBrands']);
    Route::get('/creators', [AdminController::class, 'listCreators']);
    Route::get('/campaigns', [AdminController::class, 'listCampaigns']);
    Route::get('/syncs', [AdminController::class, 'listSyncs']);

    // Verification toggles
    Route::put('/brands/{id}/verify', [AdminController::class, 'toggleBrandVerification']);
    Route::put('/creators/{id}/verify', [AdminController::class, 'toggleCreatorVerification']);
    Route::put('/creators/{id}/featured', [AdminController::class, 'toggleCreatorFeatured']);
    Route::put('/campaigns/{id}/featured', [AdminController::class, 'toggleCampaignFeatured']);

    // Admin user management (super_admin only)
    Route::get('/admins', [AdminController::class, 'listAdmins']);
    Route::post('/admins', [AdminController::class, 'createAdmin']);
    Route::put('/admins/{id}', [AdminController::class, 'updateAdmin']);
    Route::delete('/admins/{id}', [AdminController::class, 'deleteAdmin']);

    // Audit & Disputes
    Route::get('/audit-logs', [AdminController::class, 'getAuditLogs']);
    Route::get('/disputes', [AdminController::class, 'getDisputes']);
    Route::put('/disputes/{id}', [AdminController::class, 'updateDispute']);

    // Settings
    Route::get('/settings', [AdminController::class, 'getSettings']);
    Route::put('/settings', [AdminController::class, 'updateSettings']);
});
