<?php

namespace Database\Seeders;

use App\Enums\AdminRole;
use App\Models\Admin;
use App\Models\BoostPackage;
use App\Models\BrandProfile;
use App\Models\Campaign;
use App\Models\Category;
use App\Models\CreatorProfile;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // Create default super admin
        Admin::firstOrCreate(
            ['email' => 'admin@altibix.com'],
            [
                'password_hash' => Hash::make('admin123'),
                'name' => 'Super Admin',
                'role' => AdminRole::SUPER_ADMIN,
                'is_active' => true,
            ]
        );

        // Create support admin
        Admin::firstOrCreate(
            ['email' => 'support@altibix.com'],
            [
                'password_hash' => Hash::make('support123'),
                'name' => 'Support Admin',
                'role' => AdminRole::SUPPORT_ADMIN,
                'is_active' => true,
            ]
        );

        // Create boost packages
        $boostPackages = [
            [
                'name' => 'Basic Boost',
                'description' => 'Get noticed by more brands with a 7-day profile boost',
                'price' => 199,
                'duration_days' => 7,
                'features' => ['Priority listing', 'Featured badge'],
            ],
            [
                'name' => 'Pro Boost',
                'description' => 'Maximum visibility with 30-day boost and analytics',
                'price' => 699,
                'duration_days' => 30,
                'features' => ['Priority listing', 'Featured badge', 'Analytics dashboard', 'Highlighted profile'],
            ],
            [
                'name' => 'Premium Boost',
                'description' => 'Ultimate visibility for 60 days with all benefits',
                'price' => 1199,
                'duration_days' => 60,
                'features' => ['Priority listing', 'Featured badge', 'Analytics dashboard', 'Highlighted profile', 'Dedicated support'],
            ],
        ];

        foreach ($boostPackages as $package) {
            BoostPackage::firstOrCreate(
                ['name' => $package['name']],
                $package
            );
        }

        // Create default categories/niches
        $categories = [
            ['name' => 'Fashion', 'slug' => 'fashion', 'icon' => '👗'],
            ['name' => 'Beauty', 'slug' => 'beauty', 'icon' => '💄'],
            ['name' => 'Tech', 'slug' => 'tech', 'icon' => '📱'],
            ['name' => 'Food', 'slug' => 'food', 'icon' => '🍳'],
            ['name' => 'Travel', 'slug' => 'travel', 'icon' => '✈️'],
            ['name' => 'Fitness', 'slug' => 'fitness', 'icon' => '💪'],
            ['name' => 'Gaming', 'slug' => 'gaming', 'icon' => '🎮'],
            ['name' => 'Education', 'slug' => 'education', 'icon' => '📚'],
            ['name' => 'Lifestyle', 'slug' => 'lifestyle', 'icon' => '🌟'],
            ['name' => 'Music', 'slug' => 'music', 'icon' => '🎵'],
            ['name' => 'Comedy', 'slug' => 'comedy', 'icon' => '😂'],
            ['name' => 'Health', 'slug' => 'health', 'icon' => '🏥'],
        ];

        foreach ($categories as $index => $category) {
            Category::firstOrCreate(
                ['slug' => $category['slug']],
                [
                    ...$category,
                    'is_active' => true,
                    'sort_order' => $index,
                ]
            );
        }

        // Create Brands
        $brands = User::factory(5)
            ->state(['role' => \App\Enums\UserRole::BRAND])
            ->has(\App\Models\BrandProfile::factory(), 'brandProfile')
            ->create();

        // Create Creators
        $creators = User::factory(10)
            ->state(['role' => \App\Enums\UserRole::CREATOR])
            ->has(\App\Models\CreatorProfile::factory(), 'creatorProfile')
            ->create();

        // Create Campaigns for each Brand
        $brands->each(function ($user) {
            $brand = $user->brandProfile;
            \App\Models\Campaign::factory(3)
                ->for($brand, 'brand')
                ->create();
        });

        $this->command->info('Database seeded successfully!');
        $this->command->info('Default admin: admin@altibix.com / admin123');
        $this->command->info('Seeded 5 Brands (with 3 campaigns each) and 10 Creators.');
    }
}
