<?php

namespace App\Models;

use App\Enums\SyncStatus;
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Sync extends Model
{
    use HasUuids;

    protected $keyType = 'string';
    public $incrementing = false;

    protected $fillable = [
        'campaign_id',
        'brand_id',
        'creator_id',
        'status',
        'agreed_amount',
        'content_types',
        'deadline',
        'submissions',
        'milestones',
        'paid_amount',
        'chat_id',
        'contract_url',
        'payment_held_date',
        'payment_released_date',
    ];

    protected function casts(): array
    {
        return [
            'status' => SyncStatus::class,
            'agreed_amount' => 'decimal:2',
            'paid_amount' => 'decimal:2',
            'content_types' => 'array',
            'submissions' => 'array',
            'milestones' => 'array',
            'deadline' => 'datetime',
            'payment_held_date' => 'datetime',
            'payment_released_date' => 'datetime',
        ];
    }

    // Relationships
    public function campaign(): BelongsTo
    {
        return $this->belongsTo(Campaign::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(BrandProfile::class, 'brand_id');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(CreatorProfile::class, 'creator_id');
    }

    public function chat(): HasOne
    {
        return $this->hasOne(Chat::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->whereNotIn('status', [
            SyncStatus::COMPLETED,
            SyncStatus::CANCELLED,
            SyncStatus::REJECTED,
        ]);
    }
}
