<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class UserController extends Controller
{
    /**
     * Get current user profile
     * GET /api/v1/users/me
     */
    public function getProfile(Request $request): JsonResponse
    {
        $user = $request->user();

        // Load appropriate profile based on role
        if ($user->isBrand()) {
            $user->load('brandProfile');
        } else {
            $user->load('creatorProfile');
        }

        return response()->json(['user' => $user]);
    }

    /**
     * Update current user profile
     * PUT /api/v1/users/me
     */
    public function updateProfile(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'nullable|email|unique:users,email,' . $request->user()->id,
        ]);

        $user = $request->user();
        $user->update($request->only(['email']));

        return response()->json(['user' => $user->fresh()]);
    }

    /**
     * Update FCM token for push notifications
     * PUT /api/v1/users/fcm-token
     */
    public function updateFcmToken(Request $request): JsonResponse
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $user = $request->user();
        $user->update(['fcm_token' => $request->fcm_token]);

        return response()->json(['success' => true]);
    }

    /**
     * Mark onboarding as complete
     * PUT /api/v1/users/onboarding-complete
     */
    public function completeOnboarding(Request $request): JsonResponse
    {
        $user = $request->user();
        $user->update(['is_onboarding_complete' => true]);

        return response()->json(['success' => true]);
    }

    /**
     * Deactivate user account
     * DELETE /api/v1/users/me
     */
    public function deactivateAccount(Request $request): JsonResponse
    {
        $user = $request->user();

        // Revoke all tokens
        $user->tokens()->delete();

        // Soft-delete or mark as inactive (you can customize this)
        // For now, we'll just revoke tokens
        // $user->delete();

        return response()->json(['success' => true, 'message' => 'Account deactivated']);
    }
}
