<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Asset;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class UploadController extends Controller
{
    /**
     * Get presigned URL for S3 upload
     * POST /api/v1/uploads/presigned-url
     */
    public function getPresignedUrl(Request $request): JsonResponse
    {
        $request->validate([
            'filename' => 'required|string',
            'content_type' => 'required|string',
            'category' => 'nullable|string|in:profile,campaign,portfolio,chat,document',
        ]);

        $user = $request->user();
        $category = $request->get('category', 'general');
        $extension = pathinfo($request->filename, PATHINFO_EXTENSION);
        $key = "{$category}/{$user->id}/" . Str::uuid() . '.' . $extension;

        try {
            // For S3
            $disk = Storage::disk('s3');
            $adapter = $disk->getAdapter();
            $client = $adapter->getClient();
            $bucket = config('filesystems.disks.s3.bucket');

            $command = $client->getCommand('PutObject', [
                'Bucket' => $bucket,
                'Key' => $key,
                'ContentType' => $request->content_type,
            ]);

            $presignedRequest = $client->createPresignedRequest($command, '+15 minutes');
            $presignedUrl = (string) $presignedRequest->getUri();

            // Create asset record
            $asset = Asset::create([
                'name' => $request->filename,
                'category' => $category,
                'url' => config('filesystems.disks.s3.url') . '/' . $key,
                'type' => $this->getAssetType($request->content_type),
                'uploaded_by' => $user->id,
            ]);

            return response()->json([
                'presignedUrl' => $presignedUrl,
                'key' => $key,
                'assetId' => $asset->id,
                'publicUrl' => $asset->url,
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to generate presigned URL'], 500);
        }
    }

    /**
     * Confirm upload completion
     * POST /api/v1/uploads/confirm
     */
    public function confirmUpload(Request $request): JsonResponse
    {
        $request->validate([
            'asset_id' => 'required|uuid|exists:assets,id',
        ]);

        $asset = Asset::find($request->asset_id);

        // Add size if provided
        if ($request->has('size')) {
            $asset->update(['size' => $request->size]);
        }

        return response()->json(['asset' => $asset]);
    }

    /**
     * Delete an asset
     * DELETE /api/v1/uploads/{id}
     */
    public function destroy(Request $request, string $id): JsonResponse
    {
        $asset = Asset::find($id);

        if (!$asset) {
            return response()->json(['error' => 'Asset not found'], 404);
        }

        // Verify ownership
        if ($asset->uploaded_by !== $request->user()->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        // Delete from S3
        try {
            $key = str_replace(config('filesystems.disks.s3.url') . '/', '', $asset->url);
            Storage::disk('s3')->delete($key);
        } catch (\Exception $e) {
            // Log error but continue with database deletion
        }

        $asset->delete();

        return response()->json(['success' => true]);
    }

    /**
     * Upload simple image (server-side S3 proxy for compatibility)
     * POST /api/v1/uploads/image
     */
    public function uploadImage(Request $request): JsonResponse
    {
        $request->validate([
            'file' => 'required|image|max:10240', // 10MB
        ]);

        $file = $request->file('file');
        $user = $request->user();
        $path = $file->storePublicly("images/{$user->id}", 's3');
        $url = config('filesystems.disks.s3.url') . '/' . $path;

        return response()->json(['url' => $url]);
    }

    /**
     * Upload admin asset (server-side S3 proxy)
     * POST /api/v1/uploads/admin/asset
     */
    public function uploadAdminAsset(Request $request): JsonResponse
    {
        $request->validate([
            'file' => 'required|file|max:51200', // 50MB
            'name' => 'required|string',
            'category' => 'nullable|string',
        ]);

        $file = $request->file('file');
        $user = $request->user();
        $category = $request->get('category', 'general');
        $extension = $file->getClientOriginalExtension();
        
        $path = $file->storePubliclyAs(
            "admin/{$category}/{$user->id}", 
            Str::uuid() . '.' . $extension, 
            's3'
        );

        $asset = Asset::create([
            'name' => $request->name,
            'category' => $category,
            'url' => config('filesystems.disks.s3.url') . '/' . $path,
            'type' => $this->getAssetType($file->getMimeType()),
            'size' => $file->getSize(),
            'uploaded_by' => $user->id,
        ]);

        return response()->json(['asset' => $asset]);
    }

    /**
     * List all assets (for admin media library)
     * GET /api/v1/uploads/admin/assets
     */
    public function listAssets(Request $request): JsonResponse
    {
        $query = Asset::query()->latest();

        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $assets = $query->paginate($request->get('limit', 20));

        // Transform to camelCase for frontend
        $data = $assets->getCollection()->transform(function ($asset) {
            return [
                'id' => $asset->id,
                'name' => $asset->name,
                'category' => $asset->category,
                'url' => $asset->url,
                'type' => $asset->type,
                'size' => $asset->size,
                'uploadedBy' => $asset->uploaded_by,
                'createdAt' => $asset->created_at->toIsoString(),
                'updatedAt' => $asset->updated_at->toIsoString(),
            ];
        });

        return response()->json([
            'assets' => $data,
            'pagination' => [
                'page' => $assets->currentPage(),
                'limit' => $assets->perPage(),
                'total' => $assets->total(),
                'totalPages' => $assets->lastPage(),
            ]
        ]);
    }
}
    private function getAssetType(string $contentType): string
    {
        if (str_starts_with($contentType, 'image/')) {
            return 'image';
        }
        if (str_starts_with($contentType, 'video/')) {
            return 'video';
        }
        return 'document';
    }
}
