<?php

namespace App\Http\Controllers\Api;

use App\Enums\SyncStatus;
use App\Http\Controllers\Controller;
use App\Models\Chat;
use App\Models\Sync;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SyncController extends Controller
{
    /**
     * List syncs for current user
     * GET /api/v1/syncs
     */
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'status' => 'string',
        ]);

        $user = $request->user();
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 10);

        $query = Sync::query();

        // Filter by user's role
        if ($user->isBrand()) {
            $query->where('brand_id', $user->brandProfile?->id);
        } else {
            $query->where('creator_id', $user->creatorProfile?->id);
        }

        // Status filter
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $total = $query->count();
        $syncs = $query
            ->with(['campaign', 'brand', 'creator'])
            ->orderByDesc('created_at')
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        return response()->json([
            'data' => $syncs,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'totalPages' => ceil($total / $limit),
            ],
        ]);
    }

    /**
     * Get sync by ID
     * GET /api/v1/syncs/{id}
     */
    public function show(Request $request, string $id): JsonResponse
    {
        $sync = Sync::with(['campaign', 'brand', 'creator', 'chat'])->find($id);

        if (!$sync) {
            return response()->json(['error' => 'Sync not found'], 404);
        }

        // Verify access
        $user = $request->user();
        $hasAccess = ($user->isBrand() && $sync->brand_id === $user->brandProfile?->id) ||
                     ($user->isCreator() && $sync->creator_id === $user->creatorProfile?->id);

        if (!$hasAccess) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        return response()->json(['sync' => $sync]);
    }

    /**
     * Update sync
     * PUT /api/v1/syncs/{id}
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $sync = Sync::find($id);

        if (!$sync) {
            return response()->json(['error' => 'Sync not found'], 404);
        }

        $sync->update($request->only([
            'agreed_amount', 'deadline', 'content_types',
        ]));

        return response()->json(['sync' => $sync->fresh()]);
    }

    /**
     * Update sync status (workflow transition)
     * PUT /api/v1/syncs/{id}/status
     */
    public function updateStatus(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'status' => 'required|string',
        ]);

        $sync = Sync::find($id);

        if (!$sync) {
            return response()->json(['error' => 'Sync not found'], 404);
        }

        $newStatus = $request->status;

        // Validate status transition
        $validTransitions = $this->getValidTransitions($sync->status);
        if (!in_array($newStatus, $validTransitions)) {
            return response()->json(['error' => 'Invalid status transition'], 400);
        }

        $sync->update(['status' => $newStatus]);

        // Create chat if transitioning to accepted
        if ($newStatus === 'accepted' && !$sync->chat_id) {
            $chat = Chat::create([
                'sync_id' => $sync->id,
                'participants' => [$sync->brand->user_id, $sync->creator->user_id],
            ]);
            $sync->update(['chat_id' => $chat->id]);
        }

        return response()->json(['sync' => $sync->fresh()]);
    }

    /**
     * Submit content for review
     * POST /api/v1/syncs/{id}/submit
     */
    public function submitContent(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'content_url' => 'required|url',
            'notes' => 'nullable|string',
        ]);

        $sync = Sync::find($id);

        if (!$sync || $sync->creator_id !== $request->user()->creatorProfile?->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        $submissions = $sync->submissions ?? [];
        $submissions[] = [
            'url' => $request->content_url,
            'notes' => $request->notes,
            'submitted_at' => now()->toISOString(),
        ];

        $sync->update([
            'submissions' => $submissions,
            'status' => SyncStatus::REVIEW,
        ]);

        return response()->json(['sync' => $sync->fresh()]);
    }

    /**
     * Approve submitted content (brand)
     * POST /api/v1/syncs/{id}/approve
     */
    public function approveContent(Request $request, string $id): JsonResponse
    {
        $sync = Sync::find($id);

        if (!$sync || $sync->brand_id !== $request->user()->brandProfile?->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        $sync->update(['status' => SyncStatus::APPROVED]);

        return response()->json(['sync' => $sync->fresh()]);
    }

    /**
     * Request revision on submitted content (brand)
     * POST /api/v1/syncs/{id}/request-revision
     */
    public function requestRevision(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'feedback' => 'required|string',
        ]);

        $sync = Sync::find($id);

        if (!$sync || $sync->brand_id !== $request->user()->brandProfile?->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        $milestones = $sync->milestones ?? [];
        $milestones[] = [
            'type' => 'revision_requested',
            'feedback' => $request->feedback,
            'timestamp' => now()->toISOString(),
        ];

        $sync->update([
            'milestones' => $milestones,
            'status' => SyncStatus::REVISION,
        ]);

        return response()->json(['sync' => $sync->fresh()]);
    }

    /**
     * Mark sync as complete and release payment
     * POST /api/v1/syncs/{id}/complete
     */
    public function complete(Request $request, string $id): JsonResponse
    {
        $sync = Sync::with(['brand', 'creator'])->find($id);

        if (!$sync || $sync->brand_id !== $request->user()->brandProfile?->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        // Update sync status
        $sync->update([
            'status' => SyncStatus::COMPLETED,
            'payment_released_date' => now(),
        ]);

        // Update campaign stats
        $sync->campaign->increment('hired_count');

        // Update creator stats
        $sync->creator->increment('total_syncs');
        if ($sync->agreed_amount) {
            $sync->creator->increment('total_earnings', $sync->agreed_amount);
            $sync->creator->increment('wallet_balance', $sync->agreed_amount);
        }

        // Update brand stats
        if ($sync->agreed_amount) {
            $sync->brand->increment('total_spent', $sync->agreed_amount);
        }
        $sync->brand->increment('total_creators_worked_with');

        return response()->json(['sync' => $sync->fresh()]);
    }

    /**
     * Get valid status transitions for a given status
     */
    private function getValidTransitions(SyncStatus|string $currentStatus): array
    {
        $status = $currentStatus instanceof SyncStatus ? $currentStatus->value : $currentStatus;

        return match ($status) {
            'applied' => ['negotiating', 'accepted', 'rejected'],
            'invited' => ['negotiating', 'accepted', 'rejected', 'cancelled'],
            'negotiating' => ['accepted', 'rejected', 'cancelled'],
            'accepted' => ['briefing', 'cancelled'],
            'briefing' => ['contentCreation', 'cancelled'],
            'contentCreation' => ['review', 'cancelled'],
            'review' => ['approved', 'revision'],
            'revision' => ['review', 'cancelled'],
            'approved' => ['publishing', 'completed'],
            'publishing' => ['published', 'completed'],
            'published' => ['completed'],
            default => [],
        };
    }
}
