<?php

namespace App\Http\Controllers\Api;

use App\Enums\TransactionStatus;
use App\Enums\TransactionType;
use App\Http\Controllers\Controller;
use App\Models\BrandProfile;
use App\Models\BrandTransaction;
use App\Models\CreatorProfile;
use App\Models\CreatorTransaction;
use App\Models\Transaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Razorpay\Api\Api as RazorpayApi;

class PaymentController extends Controller
{
    private RazorpayApi $razorpay;

    public function __construct()
    {
        $this->razorpay = new RazorpayApi(
            config('services.razorpay.key'),
            config('services.razorpay.secret')
        );
    }

    /**
     * Create a Razorpay order for deposit
     * POST /api/v1/payments/create-order
     */
    public function createOrder(Request $request): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
        ]);

        $user = $request->user();
        $amountInPaise = (int) ($request->amount * 100);

        try {
            $order = $this->razorpay->order->create([
                'amount' => $amountInPaise,
                'currency' => 'INR',
                'receipt' => 'rcpt_' . uniqid(),
                'notes' => [
                    'user_id' => $user->id,
                    'type' => 'deposit',
                ],
            ]);

            // Create pending transaction
            Transaction::create([
                'user_id' => $user->id,
                'profile_id' => $user->isBrand() ? $user->brandProfile?->id : $user->creatorProfile?->id,
                'profile_type' => $user->role,
                'amount' => $request->amount,
                'is_credit' => true,
                'type' => TransactionType::DEPOSIT,
                'status' => TransactionStatus::PENDING,
                'razorpay_order_id' => $order->id,
            ]);

            return response()->json([
                'orderId' => $order->id,
                'amount' => $request->amount,
                'currency' => 'INR',
                'key' => config('services.razorpay.key'),
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to create payment order'], 500);
        }
    }

    /**
     * Verify Razorpay payment
     * POST /api/v1/payments/verify
     */
    public function verifyPayment(Request $request): JsonResponse
    {
        $request->validate([
            'razorpay_order_id' => 'required|string',
            'razorpay_payment_id' => 'required|string',
            'razorpay_signature' => 'required|string',
        ]);

        try {
            // Verify signature
            $attributes = [
                'razorpay_order_id' => $request->razorpay_order_id,
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature,
            ];

            $this->razorpay->utility->verifyPaymentSignature($attributes);

            // Update transaction
            $transaction = Transaction::where('razorpay_order_id', $request->razorpay_order_id)->first();

            if (!$transaction) {
                return response()->json(['error' => 'Transaction not found'], 404);
            }

            $transaction->update([
                'status' => TransactionStatus::COMPLETED,
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature,
            ]);

            // Update wallet balance
            $user = $transaction->user;
            if ($user->isBrand()) {
                $user->brandProfile?->increment('wallet_balance', $transaction->amount);

                BrandTransaction::create([
                    'brand_id' => $user->brandProfile->id,
                    'amount' => $transaction->amount,
                    'is_credit' => true,
                    'type' => TransactionType::DEPOSIT,
                    'status' => TransactionStatus::COMPLETED,
                    'reference_id' => $transaction->id,
                    'description' => 'Wallet deposit',
                ]);
            } else {
                $user->creatorProfile?->increment('wallet_balance', $transaction->amount);

                CreatorTransaction::create([
                    'creator_id' => $user->creatorProfile->id,
                    'amount' => $transaction->amount,
                    'is_credit' => true,
                    'type' => TransactionType::DEPOSIT,
                    'status' => TransactionStatus::COMPLETED,
                    'reference_id' => $transaction->id,
                    'description' => 'Wallet deposit',
                ]);
            }

            return response()->json([
                'success' => true,
                'transaction' => $transaction->fresh(),
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Payment verification failed'], 400);
        }
    }

    /**
     * Razorpay webhook handler
     * POST /api/v1/payments/webhook
     */
    public function handleWebhook(Request $request): JsonResponse
    {
        $webhookSecret = config('services.razorpay.webhook_secret');
        $signature = $request->header('X-Razorpay-Signature');

        try {
            $this->razorpay->utility->verifyWebhookSignature(
                $request->getContent(),
                $signature,
                $webhookSecret
            );

            $event = $request->input('event');
            $payload = $request->input('payload.payment.entity');

            if ($event === 'payment.captured') {
                // Handle successful payment
                $orderId = $payload['order_id'];
                $transaction = Transaction::where('razorpay_order_id', $orderId)->first();

                if ($transaction && $transaction->status === TransactionStatus::PENDING) {
                    $transaction->update([
                        'status' => TransactionStatus::COMPLETED,
                        'razorpay_payment_id' => $payload['id'],
                    ]);
                }
            }

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Webhook verification failed'], 400);
        }
    }

    /**
     * List all transactions (admin)
     * GET /api/v1/payments/admin/transactions
     */
    public function listAllTransactions(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'status' => 'string',
            'type' => 'string',
        ]);

        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = Transaction::query();

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        $total = $query->count();
        $transactions = $query
            ->with('user')
            ->orderByDesc('created_at')
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        return response()->json([
            'data' => $transactions,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'totalPages' => ceil($total / $limit),
            ],
        ]);
    }

    /**
     * Get escrow summary (admin)
     * GET /api/v1/payments/admin/escrow
     */
    public function getEscrowSummary(): JsonResponse
    {
        $heldAmount = Transaction::where('type', TransactionType::ESCROW_DEPOSIT)
            ->where('status', TransactionStatus::COMPLETED)
            ->sum('amount');

        $releasedAmount = Transaction::where('type', TransactionType::ESCROW_RELEASE)
            ->where('status', TransactionStatus::COMPLETED)
            ->sum('amount');

        return response()->json([
            'totalHeld' => $heldAmount - $releasedAmount,
            'totalReleased' => $releasedAmount,
            'pendingReleases' => Transaction::where('type', TransactionType::ESCROW_RELEASE)
                ->where('status', TransactionStatus::PENDING)
                ->count(),
        ]);
    }

    /**
     * Adjust brand wallet balance (admin)
     * POST /api/v1/payments/admin/brands/{id}/adjust-balance
     */
    public function adjustBrandBalance(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric',
            'reason' => 'required|string',
        ]);

        $brand = BrandProfile::find($id);

        if (!$brand) {
            return response()->json(['error' => 'Brand not found'], 404);
        }

        $isCredit = $request->amount > 0;
        $amount = abs($request->amount);

        if ($isCredit) {
            $brand->increment('wallet_balance', $amount);
        } else {
            $brand->decrement('wallet_balance', $amount);
        }

        BrandTransaction::create([
            'brand_id' => $brand->id,
            'amount' => $amount,
            'is_credit' => $isCredit,
            'type' => TransactionType::REFUND,
            'status' => TransactionStatus::COMPLETED,
            'description' => $request->reason,
        ]);

        return response()->json(['brand' => $brand->fresh()]);
    }

    /**
     * Adjust creator wallet balance (admin)
     * POST /api/v1/payments/admin/creators/{id}/adjust-balance
     */
    public function adjustCreatorBalance(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric',
            'reason' => 'required|string',
        ]);

        $creator = CreatorProfile::find($id);

        if (!$creator) {
            return response()->json(['error' => 'Creator not found'], 404);
        }

        $isCredit = $request->amount > 0;
        $amount = abs($request->amount);

        if ($isCredit) {
            $creator->increment('wallet_balance', $amount);
        } else {
            $creator->decrement('wallet_balance', $amount);
        }

        CreatorTransaction::create([
            'creator_id' => $creator->id,
            'amount' => $amount,
            'is_credit' => $isCredit,
            'type' => TransactionType::REFUND,
            'status' => TransactionStatus::COMPLETED,
            'description' => $request->reason,
        ]);

        return response()->json(['creator' => $creator->fresh()]);
    }
}
