<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\BrandNotification;
use App\Models\CreatorNotification;
use App\Models\Notification;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    /**
     * List notifications for current user
     * GET /api/v1/notifications
     */
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'unread' => 'nullable|boolean',
        ]);

        $user = $request->user();
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        // Use role-specific notification table
        if ($user->isBrand() && $user->brandProfile) {
            $query = BrandNotification::where('brand_id', $user->brandProfile->id);
        } elseif ($user->isCreator() && $user->creatorProfile) {
            $query = CreatorNotification::where('creator_id', $user->creatorProfile->id);
        } else {
            $query = Notification::where('user_id', $user->id);
        }

        if ($request->boolean('unread')) {
            $query->where('is_read', false);
        }

        $total = $query->count();
        $notifications = $query
            ->orderByDesc('created_at')
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        return response()->json([
            'data' => $notifications,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'totalPages' => ceil($total / $limit),
            ],
        ]);
    }

    /**
     * Get unread notification count
     * GET /api/v1/notifications/unread-count
     */
    public function getUnreadCount(Request $request): JsonResponse
    {
        $user = $request->user();

        if ($user->isBrand() && $user->brandProfile) {
            $count = BrandNotification::where('brand_id', $user->brandProfile->id)
                ->where('is_read', false)
                ->count();
        } elseif ($user->isCreator() && $user->creatorProfile) {
            $count = CreatorNotification::where('creator_id', $user->creatorProfile->id)
                ->where('is_read', false)
                ->count();
        } else {
            $count = Notification::where('user_id', $user->id)
                ->where('is_read', false)
                ->count();
        }

        return response()->json(['count' => $count]);
    }

    /**
     * Mark notification as read
     * PUT /api/v1/notifications/{id}/read
     */
    public function markAsRead(Request $request, string $id): JsonResponse
    {
        $user = $request->user();

        // Try to find in role-specific table first
        if ($user->isBrand() && $user->brandProfile) {
            $notification = BrandNotification::where('brand_id', $user->brandProfile->id)
                ->where('id', $id)
                ->first();
        } elseif ($user->isCreator() && $user->creatorProfile) {
            $notification = CreatorNotification::where('creator_id', $user->creatorProfile->id)
                ->where('id', $id)
                ->first();
        } else {
            $notification = Notification::where('user_id', $user->id)
                ->where('id', $id)
                ->first();
        }

        if (!$notification) {
            return response()->json(['error' => 'Notification not found'], 404);
        }

        $notification->update(['is_read' => true]);

        return response()->json(['success' => true]);
    }

    /**
     * Mark all notifications as read
     * PUT /api/v1/notifications/read-all
     */
    public function markAllAsRead(Request $request): JsonResponse
    {
        $user = $request->user();

        if ($user->isBrand() && $user->brandProfile) {
            BrandNotification::where('brand_id', $user->brandProfile->id)
                ->where('is_read', false)
                ->update(['is_read' => true]);
        } elseif ($user->isCreator() && $user->creatorProfile) {
            CreatorNotification::where('creator_id', $user->creatorProfile->id)
                ->where('is_read', false)
                ->update(['is_read' => true]);
        } else {
            Notification::where('user_id', $user->id)
                ->where('is_read', false)
                ->update(['is_read' => true]);
        }

        return response()->json(['success' => true]);
    }
}
