<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\FeedLike;
use App\Models\FeedPost;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class FeedController extends Controller
{
    /**
     * List feed posts
     * GET /api/v1/feed
     */
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'type' => 'nullable|string',
        ]);

        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = FeedPost::visible();

        if ($request->has('type')) {
            $query->where('post_type', $request->type);
        }

        $total = $query->count();
        $posts = $query
            ->orderByDesc('created_at')
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        // Add liked status for authenticated user
        if ($request->user()) {
            $likedPostIds = FeedLike::where('user_id', $request->user()->id)
                ->whereIn('post_id', $posts->pluck('id'))
                ->pluck('post_id')
                ->toArray();

            $posts = $posts->map(function ($post) use ($likedPostIds) {
                $post->is_liked = in_array($post->id, $likedPostIds);
                return $post;
            });
        }

        return response()->json([
            'data' => $posts,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'totalPages' => ceil($total / $limit),
            ],
        ]);
    }

    /**
     * Get single feed post
     * GET /api/v1/feed/{id}
     */
    public function show(string $id): JsonResponse
    {
        $post = FeedPost::find($id);

        if (!$post || !$post->is_visible) {
            return response()->json(['error' => 'Post not found'], 404);
        }

        return response()->json(['post' => $post]);
    }

    /**
     * Create a feed post
     * POST /api/v1/feed
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'post_type' => 'required|string',
            'content' => 'nullable|string',
            'media_urls' => 'array',
            'media_urls.*' => 'url',
        ]);

        $user = $request->user();

        $postData = [
            'post_type' => $request->post_type,
            'content' => $request->content,
            'media_urls' => $request->get('media_urls', []),
        ];

        if ($user->isBrand()) {
            $postData['brand_id'] = $user->brandProfile?->id;
        } else {
            $postData['creator_id'] = $user->creatorProfile?->id;
        }

        $post = FeedPost::create($postData);

        return response()->json(['post' => $post], 201);
    }

    /**
     * Like a post
     * POST /api/v1/feed/{id}/like
     */
    public function like(Request $request, string $id): JsonResponse
    {
        $post = FeedPost::find($id);

        if (!$post) {
            return response()->json(['error' => 'Post not found'], 404);
        }

        $userId = $request->user()->id;

        // Check if already liked
        $existingLike = FeedLike::where('post_id', $id)
            ->where('user_id', $userId)
            ->first();

        if ($existingLike) {
            return response()->json(['error' => 'Already liked'], 409);
        }

        FeedLike::create([
            'post_id' => $id,
            'user_id' => $userId,
        ]);

        $post->increment('like_count');

        return response()->json(['success' => true, 'like_count' => $post->fresh()->like_count]);
    }

    /**
     * Unlike a post
     * DELETE /api/v1/feed/{id}/like
     */
    public function unlike(Request $request, string $id): JsonResponse
    {
        $post = FeedPost::find($id);

        if (!$post) {
            return response()->json(['error' => 'Post not found'], 404);
        }

        $userId = $request->user()->id;

        $like = FeedLike::where('post_id', $id)
            ->where('user_id', $userId)
            ->first();

        if (!$like) {
            return response()->json(['error' => 'Not liked'], 404);
        }

        $like->delete();
        $post->decrement('like_count');

        return response()->json(['success' => true, 'like_count' => max(0, $post->fresh()->like_count)]);
    }

    /**
     * Delete a post
     * DELETE /api/v1/feed/{id}
     */
    public function destroy(Request $request, string $id): JsonResponse
    {
        $post = FeedPost::find($id);

        if (!$post) {
            return response()->json(['error' => 'Post not found'], 404);
        }

        $user = $request->user();

        // Verify ownership
        $isOwner = ($user->isBrand() && $post->brand_id === $user->brandProfile?->id) ||
                   ($user->isCreator() && $post->creator_id === $user->creatorProfile?->id);

        if (!$isOwner) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        // Delete likes
        FeedLike::where('post_id', $id)->delete();

        $post->delete();

        return response()->json(['success' => true]);
    }
}
