<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CreatorProfile;
use App\Models\CreatorTransaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CreatorController extends Controller
{
    /**
     * Get current creator profile
     * GET /api/v1/creators/profile
     */
    public function getProfile(Request $request): JsonResponse
    {
        $creator = $request->user()->creatorProfile()->with('user')->first();

        if (!$creator) {
            return response()->json(['error' => 'Creator profile not found'], 404);
        }

        return (new \App\Http\Resources\CreatorResource($creator))->response();
    }

    // ... (updateProfile, updatePortfolio, updateRateCard, getWallet, getTransactions)

    /**
     * Get creator by ID (public)
     * GET /api/v1/creators/{id}
     */
    public function show(string $id): JsonResponse
    {
        $creator = CreatorProfile::find($id);

        if (!$creator) {
            return response()->json(['error' => 'Creator not found'], 404);
        }

        return (new \App\Http\Resources\CreatorResource($creator))->response();
    }

    /**
     * List creators (public)
     * GET /api/v1/creators
     */
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'niches' => 'nullable|string',
            'verified' => 'nullable|boolean',
            'top' => 'nullable|boolean',
        ]);

        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = CreatorProfile::query();

        if ($request->has('niches')) {
            $niches = explode(',', $request->niches);
            $query->where(function ($q) use ($niches) {
                foreach ($niches as $niche) {
                    $q->orWhereJsonContains('niches', trim($niche));
                }
            });
        }

        if ($request->has('verified')) {
            $query->where('is_verified', $request->boolean('verified'));
        }

        if ($request->boolean('top')) {
            $query->where('is_top_creator', true);
        }

        $creators = $query
            ->orderByDesc('rating')
            ->orderByDesc('total_syncs')
            ->paginate($limit);

        return \App\Http\Resources\CreatorResource::collection($creators)->response();
    }

    /**
     * Get featured/top creators
     * GET /api/v1/creators/featured
     */
    public function featured(): JsonResponse
    {
        $creators = CreatorProfile::where('is_top_creator', true)
            ->orWhere('is_verified', true)
            ->orderByDesc('rating')
            ->take(10)
            ->get();

        return response()->json(['creators' => $creators]);
    }

    /**
     * Calculate profile completion percentage
     */
    private function calculateProfileCompletion(CreatorProfile $creator): int
    {
        $fields = [
            'display_name' => 15,
            'bio' => 15,
            'profile_image_url' => 15,
            'location' => 5,
            'niches' => 15,
            'social_links' => 10,
            'rate_card' => 15,
            'portfolio' => 10,
        ];

        $completion = 0;
        foreach ($fields as $field => $weight) {
            $value = $creator->$field;
            if (!empty($value) && (is_array($value) ? count($value) > 0 : true)) {
                $completion += $weight;
            }
        }

        return min(100, $completion);
    }
}
