<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Chat;
use App\Models\Message;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ChatController extends Controller
{
    /**
     * List all chats for current user
     * GET /api/v1/chats
     */
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();

        $chats = Chat::whereJsonContains('participants', $user->id)
            ->with(['latestMessage'])
            ->orderByDesc('last_message_at')
            ->get();

        // Add sync info for context
        $chats->load('sync.campaign', 'sync.brand', 'sync.creator');

        return response()->json(['chats' => $chats]);
    }

    /**
     * Get chat by ID
     * GET /api/v1/chats/{id}
     */
    public function show(Request $request, string $id): JsonResponse
    {
        $chat = Chat::with(['sync.campaign', 'sync.brand', 'sync.creator'])->find($id);

        if (!$chat) {
            return response()->json(['error' => 'Chat not found'], 404);
        }

        // Verify user is participant
        if (!in_array($request->user()->id, $chat->participants ?? [])) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        return response()->json(['chat' => $chat]);
    }

    /**
     * Get messages for a chat
     * GET /api/v1/chats/{id}/messages
     */
    public function getMessages(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
        ]);

        $chat = Chat::find($id);

        if (!$chat) {
            return response()->json(['error' => 'Chat not found'], 404);
        }

        // Verify user is participant
        if (!in_array($request->user()->id, $chat->participants ?? [])) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 50);

        $query = Message::where('chat_id', $id);

        $total = $query->count();
        $messages = $query
            ->with('sender')
            ->orderByDesc('created_at')
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        return response()->json([
            'data' => $messages,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'totalPages' => ceil($total / $limit),
            ],
        ]);
    }

    /**
     * Send a message
     * POST /api/v1/chats/{id}/messages
     */
    public function sendMessage(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'content' => 'required_without:attachments|string',
            'message_type' => 'in:text,image,file,system',
            'attachments' => 'array',
        ]);

        $chat = Chat::find($id);

        if (!$chat) {
            return response()->json(['error' => 'Chat not found'], 404);
        }

        // Verify user is participant
        $userId = $request->user()->id;
        if (!in_array($userId, $chat->participants ?? [])) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        $message = Message::create([
            'chat_id' => $id,
            'sender_id' => $userId,
            'content' => $request->content,
            'message_type' => $request->get('message_type', 'text'),
            'attachments' => $request->get('attachments', []),
            'read_by' => [$userId],
        ]);

        // Update chat last message time
        $chat->update(['last_message_at' => now()]);

        // Broadcast event for real-time updates
        broadcast(new \App\Events\MessageSent($message))->toOthers();

        return response()->json(['message' => $message->load('sender')], 201);
    }

    /**
     * Mark messages as read
     * PUT /api/v1/chats/{id}/read
     */
    public function markAsRead(Request $request, string $id): JsonResponse
    {
        $chat = Chat::find($id);

        if (!$chat) {
            return response()->json(['error' => 'Chat not found'], 404);
        }

        $userId = $request->user()->id;
        if (!in_array($userId, $chat->participants ?? [])) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        // Mark all unread messages as read
        Message::where('chat_id', $id)
            ->whereJsonDoesntContain('read_by', $userId)
            ->get()
            ->each(function ($message) use ($userId) {
                $readBy = $message->read_by ?? [];
                $readBy[] = $userId;
                $message->update(['read_by' => array_unique($readBy)]);
            });

        return response()->json(['success' => true]);
    }
}
