<?php

namespace App\Http\Controllers\Api;

use App\Enums\CampaignStatus;
use App\Http\Controllers\Controller;
use App\Models\Campaign;
use App\Models\Sync;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CampaignController extends Controller
{
    /**
     * List all campaigns with pagination and filters
     * GET /api/v1/campaigns
     */
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'sortBy' => 'string|in:createdAt,created_at,budget,title',
            'sortOrder' => 'string|in:asc,desc',
            'status' => 'string|in:draft,active,paused,completed,cancelled',
            'niches' => 'string',
            'contentTypes' => 'string',
        ]);

        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 10);
        $sortBy = $request->get('sortBy', 'created_at');
        $sortOrder = $request->get('sortOrder', 'desc');

        $query = Campaign::query();

        // Status filter
        if ($request->has('status')) {
            $query->where('status', $request->status);
        } else {
            $query->where('status', CampaignStatus::ACTIVE);
        }

        // ... (niches and contentTypes filters omitted for brevity but should be kept if replacing partially, here replacing full method for cleanliness)
        if ($request->has('niches')) {
            $niches = explode(',', $request->niches);
            $query->where(function ($q) use ($niches) {
                foreach ($niches as $niche) {
                    $q->orWhereJsonContains('target_niches', trim($niche));
                }
            });
        }

        if ($request->has('contentTypes')) {
            $types = explode(',', $request->contentTypes);
            $query->where(function ($q) use ($types) {
                foreach ($types as $type) {
                    $q->orWhereJsonContains('content_types', trim($type));
                }
            });
        }

        $campaigns = $query
            ->with(['brand.user'])
            ->orderBy($sortBy === 'createdAt' ? 'created_at' : $sortBy, $sortOrder)
            ->paginate($limit);

        return \App\Http\Resources\CampaignResource::collection($campaigns)->response();
    }

    /**
     * Get featured campaigns
     * GET /api/v1/campaigns/featured
     */
    public function featured(): JsonResponse
    {
        $campaigns = Campaign::where('status', CampaignStatus::ACTIVE)
            ->where(function ($q) {
                $q->where('is_featured', true)->orWhere('is_hot', true);
            })
            ->with(['brand.user'])
            ->orderByDesc('created_at')
            ->take(10)
            ->get();
        
        // Note: For custom collections without pagination, we can still use Resource::collection
        return \App\Http\Resources\CampaignResource::collection($campaigns)->response();
    }

    /**
     * Get campaign by ID
     * GET /api/v1/campaigns/{id}
     */
    public function show(string $id): JsonResponse
    {
        $campaign = Campaign::with([
            'brand',
            'syncs' => fn($q) => $q->take(10)->orderByDesc('created_at')->with('creator:id,display_name,profile_image_url'),
        ])->find($id);

        if (!$campaign) {
            return response()->json(['error' => 'Campaign not found'], 404);
        }

        return (new \App\Http\Resources\CampaignResource($campaign))->response();
    }

    /**
     * Create a new campaign
     * POST /api/v1/campaigns
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'cover_image' => 'nullable|string|url',
            'content_types' => 'array',
            'target_niches' => 'array',
            'budget' => 'nullable|numeric|min:0',
            'payment_type' => 'in:fixed,negotiable',
            'max_creators' => 'nullable|integer|min:1',
            'deadline' => 'nullable|date',
            'status' => 'in:draft,active',
            'requirements' => 'array',
        ]);

        $user = $request->user();
        $brand = $user->brandProfile;

        if (!$brand) {
            return response()->json(['error' => 'Brand profile not found'], 400);
        }

        $campaign = Campaign::create([
            'brand_id' => $brand->id,
            'brand_name' => $brand->company_name,
            'brand_logo' => $brand->logo,
            ...$request->only([
                'title', 'description', 'cover_image', 'content_types',
                'target_niches', 'budget', 'payment_type', 'max_creators',
                'deadline', 'status', 'requirements', 'deliverables',
            ]),
        ]);

        // Update brand stats
        $brand->increment('total_campaigns');
        if ($campaign->status === CampaignStatus::ACTIVE) {
            $brand->increment('active_campaigns');
        }

        return response()->json(['campaign' => $campaign], 201);
    }

    /**
     * Update campaign
     * PUT /api/v1/campaigns/{id}
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $campaign = Campaign::with('brand')->find($id);

        if (!$campaign || $campaign->brand->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Not authorized to update this campaign'], 403);
        }

        $campaign->update($request->only([
            'title', 'description', 'cover_image', 'content_types',
            'target_niches', 'budget', 'payment_type', 'max_creators',
            'deadline', 'status', 'requirements', 'deliverables',
            'is_featured', 'is_hot',
        ]));

        return response()->json(['campaign' => $campaign->fresh()]);
    }

    /**
     * Delete campaign
     * DELETE /api/v1/campaigns/{id}
     */
    public function destroy(Request $request, string $id): JsonResponse
    {
        $campaign = Campaign::with('brand')->find($id);

        if (!$campaign || $campaign->brand->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Not authorized to delete this campaign'], 403);
        }

        // Check for active syncs
        $activeSyncs = Sync::where('campaign_id', $id)
            ->whereNotIn('status', ['completed', 'cancelled', 'rejected'])
            ->count();

        if ($activeSyncs > 0) {
            return response()->json(['error' => 'Cannot delete campaign with active collaborations'], 400);
        }

        $campaign->delete();

        return response()->json(['success' => true]);
    }

    /**
     * Apply to a campaign (creator)
     * POST /api/v1/campaigns/{id}/apply
     */
    public function apply(Request $request, string $id): JsonResponse
    {
        $user = $request->user();
        $creator = $user->creatorProfile;

        if (!$creator) {
            return response()->json(['error' => 'Creator profile not found'], 400);
        }

        $campaign = Campaign::find($id);

        if (!$campaign || $campaign->status !== CampaignStatus::ACTIVE) {
            return response()->json(['error' => 'Campaign not available for applications'], 400);
        }

        // Check if already applied
        $existingSync = Sync::where('campaign_id', $id)
            ->where('creator_id', $creator->id)
            ->first();

        if ($existingSync) {
            return response()->json(['error' => 'Already applied to this campaign'], 409);
        }

        // Check max creators
        if ($campaign->max_creators && $campaign->hired_count >= $campaign->max_creators) {
            return response()->json(['error' => 'Campaign has reached maximum creators'], 400);
        }

        $sync = Sync::create([
            'campaign_id' => $id,
            'brand_id' => $campaign->brand_id,
            'creator_id' => $creator->id,
            'status' => 'applied',
            'content_types' => $campaign->content_types,
        ]);

        $campaign->increment('applications_count');

        return response()->json(['sync' => $sync], 201);
    }

    /**
     * Get campaign applications (brand)
     * GET /api/v1/campaigns/{id}/applications
     */
    public function applications(Request $request, string $id): JsonResponse
    {
        $campaign = Campaign::with('brand')->find($id);

        if (!$campaign || $campaign->brand->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        $applications = Sync::where('campaign_id', $id)
            ->with('creator')
            ->orderByDesc('created_at')
            ->get();

        return response()->json(['applications' => $applications]);
    }

    /**
     * Invite a creator to campaign (brand)
     * POST /api/v1/campaigns/{id}/invite
     */
    public function invite(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'creatorId' => 'required|uuid|exists:creator_profiles,id',
        ]);

        $campaign = Campaign::with('brand')->find($id);

        if (!$campaign || $campaign->brand->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Not authorized'], 403);
        }

        // Check if already invited/applied
        $existingSync = Sync::where('campaign_id', $id)
            ->where('creator_id', $request->creatorId)
            ->first();

        if ($existingSync) {
            return response()->json(['error' => 'Creator already has a sync for this campaign'], 409);
        }

        $sync = Sync::create([
            'campaign_id' => $id,
            'brand_id' => $campaign->brand_id,
            'creator_id' => $request->creatorId,
            'status' => 'invited',
            'content_types' => $campaign->content_types,
        ]);

        return response()->json(['sync' => $sync], 201);
    }
}
