<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\BrandProfile;
use App\Models\BrandTransaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class BrandController extends Controller
{
    /**
     * Get current brand profile
     * GET /api/v1/brands/profile
     */
    public function getProfile(Request $request): JsonResponse
    {
        $brand = $request->user()->brandProfile()->with('user')->first();

        if (!$brand) {
            return response()->json(['error' => 'Brand profile not found'], 404);
        }

        return (new \App\Http\Resources\BrandResource($brand))->response();
    }

    // ... (updateProfile, getWallet, getTransactions remain unchanged for now or need simpler updates)

    /**
     * Get brand by ID (public)
     * GET /api/v1/brands/{id}
     */
    public function show(string $id): JsonResponse
    {
        $brand = BrandProfile::with('campaigns')->find($id);

        if (!$brand) {
            return response()->json(['error' => 'Brand not found'], 404);
        }

        return (new \App\Http\Resources\BrandResource($brand))->response();
    }

    /**
     * List all brands (public)
     * GET /api/v1/brands
     */
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'page' => 'integer|min:1',
            'limit' => 'integer|min:1|max:100',
            'industry' => 'nullable|string',
            'verified' => 'nullable|boolean',
        ]);

        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = BrandProfile::query()->with('user');

        if ($request->has('industry')) {
            $query->where('industry', $request->industry);
        }

        if ($request->has('verified')) {
            $query->where('is_verified', $request->boolean('verified'));
        }

        $brands = $query
            ->orderByDesc('created_at')
            ->paginate($limit);

        return \App\Http\Resources\BrandResource::collection($brands)->response();
    }

    /**
     * Calculate profile completion percentage
     */
    private function calculateProfileCompletion(BrandProfile $brand): int
    {
        $fields = [
            'company_name' => 15,
            'logo' => 15,
            'website' => 10,
            'description' => 15,
            'industry' => 10,
            'target_niches' => 10,
            'contact_email' => 10,
            'contact_phone' => 5,
            'location' => 10,
        ];

        $completion = 0;
        foreach ($fields as $field => $weight) {
            $value = $brand->$field;
            if (!empty($value) && (is_array($value) ? count($value) > 0 : true)) {
                $completion += $weight;
            }
        }

        return min(100, $completion);
    }
}
