<?php

namespace App\Http\Controllers\Api;

use App\Enums\TransactionStatus;
use App\Enums\TransactionType;
use App\Http\Controllers\Controller;
use App\Models\ActiveBoost;
use App\Models\BoostPackage;
use App\Models\Transaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class BoostController extends Controller
{
    /**
     * List all boost packages
     * GET /api/v1/boosts/packages
     */
    public function listPackages(): JsonResponse
    {
        $packages = BoostPackage::active()->orderBy('price')->get();

        return response()->json(['packages' => $packages]);
    }

    /**
     * Get boost package by ID
     * GET /api/v1/boosts/packages/{id}
     */
    public function showPackage(string $id): JsonResponse
    {
        $package = BoostPackage::find($id);

        if (!$package) {
            return response()->json(['error' => 'Package not found'], 404);
        }

        return response()->json(['package' => $package]);
    }

    /**
     * Purchase a boost
     * POST /api/v1/boosts/purchase
     */
    public function purchase(Request $request): JsonResponse
    {
        $request->validate([
            'package_id' => 'required|uuid|exists:boost_packages,id',
        ]);

        $user = $request->user();
        $package = BoostPackage::find($request->package_id);

        if (!$package || !$package->is_active) {
            return response()->json(['error' => 'Package not available'], 400);
        }

        // Check wallet balance (for creators)
        if ($user->isCreator()) {
            $creator = $user->creatorProfile;
            if (!$creator || $creator->wallet_balance < $package->price) {
                return response()->json(['error' => 'Insufficient wallet balance'], 400);
            }

            // Deduct from wallet
            $creator->decrement('wallet_balance', $package->price);

            // Create transaction
            Transaction::create([
                'user_id' => $user->id,
                'profile_id' => $creator->id,
                'profile_type' => 'creator',
                'amount' => $package->price,
                'is_credit' => false,
                'type' => TransactionType::BOOST_PURCHASE,
                'status' => TransactionStatus::COMPLETED,
                'reference_id' => $package->id,
            ]);
        }

        // Create active boost
        $boost = ActiveBoost::create([
            'user_id' => $user->id,
            'package_id' => $package->id,
            'start_date' => now(),
            'end_date' => now()->addDays($package->duration_days),
            'is_active' => true,
        ]);

        return response()->json(['boost' => $boost->load('package')], 201);
    }

    /**
     * Get active boosts for current user
     * GET /api/v1/boosts/active
     */
    public function getActiveBoosts(Request $request): JsonResponse
    {
        $user = $request->user();

        $boosts = ActiveBoost::where('user_id', $user->id)
            ->where('is_active', true)
            ->where('end_date', '>', now())
            ->with('package')
            ->get();

        return response()->json(['boosts' => $boosts]);
    }

    /**
     * Get boost history
     * GET /api/v1/boosts/history
     */
    public function getHistory(Request $request): JsonResponse
    {
        $user = $request->user();

        $boosts = ActiveBoost::where('user_id', $user->id)
            ->with('package')
            ->orderByDesc('created_at')
            ->get();

        return response()->json(['boosts' => $boosts]);
    }
}
