<?php

namespace App\Http\Controllers\Api;

use App\Enums\UserRole;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\BrandProfile;
use App\Models\CreatorProfile;
use App\Models\RefreshToken;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    /**
     * Request OTP for phone login
     * POST /api/v1/auth/request-otp
     */
    public function requestOtp(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string|min:10|max:15',
        ]);

        // In production, integrate with SMS provider (Twilio/MSG91)
        // For now, we'll simulate OTP generation
        $otp = rand(100000, 999999);

        // Store OTP in cache (valid for 5 minutes)
        cache()->put('otp:' . $request->phone, $otp, now()->addMinutes(5));

        // In production, send SMS here
        // For development, return OTP in response
        return response()->json([
            'message' => 'OTP sent successfully',
            'debug_otp' => app()->environment('local') ? $otp : null,
        ]);
    }

    /**
     * Verify OTP and login/register
     * POST /api/v1/auth/verify-otp
     */
    public function verifyOtp(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string|min:10|max:15',
            'otp' => 'required|string|size:6',
        ]);

        $cachedOtp = cache()->get('otp:' . $request->phone);

        if (!$cachedOtp || $cachedOtp != $request->otp) {
            return response()->json(['error' => 'Invalid or expired OTP'], 401);
        }

        // Clear OTP from cache
        cache()->forget('otp:' . $request->phone);

        // Find or create user
        $user = User::where('phone', $request->phone)->first();
        $isNewUser = false;

        if (!$user) {
            $isNewUser = true;
            $user = User::create([
                'phone' => $request->phone,
                'role' => UserRole::CREATOR,
            ]);
        }

        // Generate tokens
        $accessToken = $user->createToken('api')->plainTextToken;
        $refreshToken = $this->generateRefreshToken($user);

        // Load profile relationship for resource transformation
        if ($user->role === UserRole::BRAND->value) {
            $user->load('brandProfile');
        } elseif ($user->role === UserRole::CREATOR->value) {
            $user->load('creatorProfile');
        }

        return response()->json([
            'accessToken' => $accessToken,
            'refreshToken' => $refreshToken,
            'user' => new \App\Http\Resources\UserResource($user),
            'isNewUser' => $isNewUser,
        ]);
    }

    /**
     * Register a new user (brand or creator)
     * POST /api/v1/auth/register
     */
    public function register(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string|unique:users,phone',
            'role' => 'required|in:brand,creator',
            'email' => 'nullable|email|unique:users,email',
            // Brand fields
            'company_name' => 'required_if:role,brand|string|max:255',
            // Creator fields
            'display_name' => 'required_if:role,creator|string|max:255',
        ]);

        $user = User::create([
            'phone' => $request->phone,
            'email' => $request->email,
            'role' => $request->role,
        ]);

        if ($request->role === 'brand') {
            BrandProfile::create([
                'user_id' => $user->id,
                'company_name' => $request->company_name,
            ]);
        } else {
            CreatorProfile::create([
                'user_id' => $user->id,
                'display_name' => $request->display_name,
            ]);
        }

        $accessToken = $user->createToken('api')->plainTextToken;
        $refreshToken = $this->generateRefreshToken($user);

        return response()->json([
            'accessToken' => $accessToken,
            'refreshToken' => $refreshToken,
            'user' => new \App\Http\Resources\UserResource($user->load($request->role === 'brand' ? 'brandProfile' : 'creatorProfile')),
        ], 201);
    }

    /**
     * Refresh access token
     * POST /api/v1/auth/refresh
     */
    public function refreshToken(Request $request): JsonResponse
    {
        $request->validate([
            'refreshToken' => 'required|string',
        ]);

        $refreshToken = RefreshToken::where('token', $request->refreshToken)->first();

        if (!$refreshToken || $refreshToken->isExpired()) {
            return response()->json(['error' => 'Invalid or expired refresh token'], 401);
        }

        $user = $refreshToken->user;

        // Revoke old tokens and generate new ones
        $refreshToken->delete();
        $user->tokens()->delete();

        $accessToken = $user->createToken('api')->plainTextToken;
        $newRefreshToken = $this->generateRefreshToken($user);

        return response()->json([
            'accessToken' => $accessToken,
            'refreshToken' => $newRefreshToken,
            'user' => $user,
        ]);
    }

    /**
     * Admin login with email/password
     * POST /api/v1/auth/admin/login
     */
    public function adminLogin(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        $admin = Admin::where('email', $request->email)->first();

        if (!$admin || !Hash::check($request->password, $admin->password_hash)) {
            return response()->json(['error' => 'Invalid credentials'], 401);
        }

        if (!$admin->is_active) {
            return response()->json(['error' => 'Account deactivated'], 403);
        }

        // Update last login
        $admin->update(['last_login' => now()]);

        $token = $admin->createToken('admin-api', ['admin'])->plainTextToken;

        return response()->json([
            'accessToken' => $token,
            'admin' => $admin,
        ]);
    }

    /**
     * Logout and invalidate refresh token
     * POST /api/v1/auth/logout
     */
    public function logout(Request $request): JsonResponse
    {
        $request->validate([
            'refreshToken' => 'nullable|string',
        ]);

        // Revoke current access token
        $request->user()?->currentAccessToken()?->delete();

        // Revoke refresh token if provided
        if ($request->refreshToken) {
            RefreshToken::where('token', $request->refreshToken)->delete();
        }

        return response()->json(['success' => true]);
    }

    /**
     * Generate a refresh token for the user
     */
    private function generateRefreshToken(User $user): string
    {
        $token = Str::random(64);

        RefreshToken::create([
            'token' => $token,
            'user_id' => $user->id,
            'expires_at' => now()->addDays(30),
        ]);

        return $token;
    }
}
