<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Resources\AdminResource;
use App\Http\Resources\AuditLogResource;
use App\Http\Resources\BrandResource;
use App\Http\Resources\CampaignResource;
use App\Http\Resources\CreatorResource;
use App\Http\Resources\DisputeResource;
use App\Http\Resources\SyncResource;
use App\Models\Admin;
use App\Models\AuditLog;
use App\Models\BrandProfile;
use App\Models\Campaign;
use App\Models\CreatorProfile;
use App\Models\Dispute;
use App\Models\Sync;
use App\Models\AppSetting;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AdminController extends Controller
{
    /**
     * Get dashboard statistics
     * GET /api/v1/admin/dashboard
     */
    public function getDashboardStats(): JsonResponse
    {
        return response()->json([
            'stats' => [
                'totalBrands' => BrandProfile::count(),
                'totalCreators' => CreatorProfile::count(),
                'totalCampaigns' => Campaign::count(),
                'activeCampaigns' => Campaign::where('status', 'active')->count(),
                'totalSyncs' => Sync::count(),
                'activeSyncs' => Sync::active()->count(),
                'pendingDisputes' => Dispute::open()->count(),
            ],
            'recentActivity' => AuditLog::with('admin')
                ->orderByDesc('performed_at')
                ->take(10)
                ->get(),
        ]);
    }

    /**
     * Get analytics data
     * GET /api/v1/admin/analytics
     */
    public function getAnalytics(Request $request): JsonResponse
    {
        $days = $request->integer('days', 30);

        return response()->json([
            'newBrands' => BrandProfile::where('created_at', '>=', now()->subDays($days))->count(),
            'newCreators' => CreatorProfile::where('created_at', '>=', now()->subDays($days))->count(),
            'newCampaigns' => Campaign::where('created_at', '>=', now()->subDays($days))->count(),
            'completedSyncs' => Sync::where('status', 'completed')
                ->where('updated_at', '>=', now()->subDays($days))
                ->count(),
        ]);
    }

    /**
     * List all brands (admin view)
     * GET /api/v1/admin/brands
     */
    public function listBrands(Request $request): JsonResponse
    {
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = BrandProfile::with('user');

        $brands = $query
            ->orderByDesc('created_at')
            ->paginate($limit);

        return BrandResource::collection($brands)->response();
    }

    /**
     * List all creators (admin view)
     * GET /api/v1/admin/creators
     */
    public function listCreators(Request $request): JsonResponse
    {
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = CreatorProfile::with('user');

        $creators = $query
            ->orderByDesc('created_at')
            ->paginate($limit);

        return CreatorResource::collection($creators)->response();
    }

    /**
     * List all campaigns (admin view)
     * GET /api/v1/admin/campaigns
     */
    public function listCampaigns(Request $request): JsonResponse
    {
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = Campaign::with('brand.user');

        $campaigns = $query
            ->orderByDesc('created_at')
            ->paginate($limit);

        return CampaignResource::collection($campaigns)->response();
    }

    /**
     * List all syncs (admin view)
     * GET /api/v1/admin/syncs
     */
    public function listSyncs(Request $request): JsonResponse
    {
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);

        $query = Sync::with(['campaign.brand.user', 'brand.user', 'creator.user']);

        $syncs = $query
            ->orderByDesc('created_at')
            ->paginate($limit);

        return SyncResource::collection($syncs)->response();
    }

    /**
     * Toggle brand verification
     * PUT /api/v1/admin/brands/{id}/verify
     */
    public function toggleBrandVerification(string $id): JsonResponse
    {
        $brand = BrandProfile::find($id);

        if (!$brand) {
            return response()->json(['error' => 'Brand not found'], 404);
        }

        $brand->update(['is_verified' => !$brand->is_verified]);

        return response()->json(['brand' => $brand->fresh()]);
    }

    /**
     * Toggle creator verification
     * PUT /api/v1/admin/creators/{id}/verify
     */
    public function toggleCreatorVerification(string $id): JsonResponse
    {
        $creator = CreatorProfile::find($id);

        if (!$creator) {
            return response()->json(['error' => 'Creator not found'], 404);
        }

        $creator->update(['is_verified' => !$creator->is_verified]);

        return response()->json(['creator' => $creator->fresh()]);
    }

    /**
     * Toggle creator featured status
     * PUT /api/v1/admin/creators/{id}/featured
     */
    public function toggleCreatorFeatured(string $id): JsonResponse
    {
        $creator = CreatorProfile::find($id);

        if (!$creator) {
            return response()->json(['error' => 'Creator not found'], 404);
        }

        $creator->update(['is_top_creator' => !$creator->is_top_creator]);

        return response()->json(['creator' => $creator->fresh()]);
    }

    /**
     * Toggle campaign featured status
     * PUT /api/v1/admin/campaigns/{id}/featured
     */
    public function toggleCampaignFeatured(string $id): JsonResponse
    {
        $campaign = Campaign::find($id);

        if (!$campaign) {
            return response()->json(['error' => 'Campaign not found'], 404);
        }

        $campaign->update(['is_featured' => !$campaign->is_featured]);

        return response()->json(['campaign' => $campaign->fresh()]);
    }

    /**
     * List all admin users
     * GET /api/v1/admin/admins
     */
    public function listAdmins(): JsonResponse
    {
        $admins = Admin::orderByDesc('created_at')->get();
        return AdminResource::collection($admins)->response();
    }

    /**
     * Create new admin user
     * POST /api/v1/admin/admins
     */
    public function createAdmin(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email|unique:admins,email',
            'password' => 'required|string|min:8',
            'name' => 'required|string|max:255',
            'role' => 'required|in:super_admin,finance_admin,support_admin,content_admin',
        ]);

        $admin = Admin::create([
            'email' => $request->email,
            'password_hash' => Hash::make($request->password),
            'name' => $request->name,
            'role' => $request->role,
        ]);

        return response()->json(['admin' => $admin], 201);
    }

    /**
     * Update admin user
     * PUT /api/v1/admin/admins/{id}
     */
    public function updateAdmin(Request $request, string $id): JsonResponse
    {
        $admin = Admin::find($id);

        if (!$admin) {
            return response()->json(['error' => 'Admin not found'], 404);
        }

        $data = $request->only(['name', 'role', 'is_active']);

        if ($request->has('password')) {
            $data['password_hash'] = Hash::make($request->password);
        }

        $admin->update($data);

        return response()->json(['admin' => $admin->fresh()]);
    }

    /**
     * Delete admin user
     * DELETE /api/v1/admin/admins/{id}
     */
    public function deleteAdmin(string $id): JsonResponse
    {
        $admin = Admin::find($id);

        if (!$admin) {
            return response()->json(['error' => 'Admin not found'], 404);
        }

        $admin->delete();

        return response()->json(['success' => true]);
    }

    /**
     * Get audit logs
     * GET /api/v1/admin/audit-logs
     */
    public function getAuditLogs(Request $request): JsonResponse
    {
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 50);

        $query = AuditLog::with('admin');

        $logs = $query
            ->orderByDesc('performed_at')
            ->paginate($limit);

        return AuditLogResource::collection($logs)->response();
    }

    /**
     * Get disputes
     * GET /api/v1/admin/disputes
     */
    public function getDisputes(Request $request): JsonResponse
    {
        $page = $request->integer('page', 1);
        $limit = $request->integer('limit', 20);
        $status = $request->get('status');

        $query = Dispute::with('sync');

        if ($status) {
            $query->where('status', $status);
        }

        $disputes = $query
            ->orderByDesc('created_at')
            ->paginate($limit);

        return DisputeResource::collection($disputes)->response();
    }

    /**
     * Update dispute status
     * PUT /api/v1/admin/disputes/{id}
     */
    public function updateDispute(Request $request, string $id): JsonResponse
    {
        $dispute = Dispute::find($id);

        if (!$dispute) {
            return response()->json(['error' => 'Dispute not found'], 404);
        }

        $dispute->update($request->only(['status', 'resolution', 'resolved_by', 'resolved_at']));

        return response()->json(['dispute' => $dispute->fresh()]);
    }

    /**
     * Get app settings
     * GET /api/v1/admin/settings
     */
    public function getSettings(): JsonResponse
    {
        $settings = AppSetting::all()->pluck('value', 'key');
        return response()->json(['settings' => $settings]);
    }

    /**
     * Update app settings
     * PUT /api/v1/admin/settings
     */
    public function updateSettings(Request $request): JsonResponse
    {
        foreach ($request->all() as $key => $value) {
            AppSetting::set($key, $value);
        }

        return response()->json(['success' => true]);
    }
}
